package com.nextbiometrics.system;

import com.nextbiometrics.system.jna.NBUInt;
import com.nextbiometrics.system.jna.ptr.NBUIntByReference;
import com.sun.jna.Native;
import java.io.IOException;
import java.util.concurrent.CancellationException;
import java.util.concurrent.TimeoutException;

public final class NBErrors {
    public static final int OK = 0;
    public static final int ERROR_FAILED = -100;
    public static final int ERROR_ARGUMENT = -200;
    public static final int ERROR_ARGUMENT_NULL = -201;
    public static final int ERROR_ARGUMENT_OUT_OF_RANGE = -202;
    public static final int ERROR_INVALID_ENUM_ARGUMENT = -203;
    public static final int ERROR_INSUFFICIENT_BUFFER = -204;
    public static final int ERROR_INDEX_OUT_OF_RANGE = -205;
    public static final int ERROR_FORMAT = -300;
    public static final int ERROR_MEMORY = -400;
    public static final int ERROR_OUT_OF_MEMORY = -401;
    public static final int ERROR_MEMORY_CORRUPTION = -402;
    public static final int ERROR_ARITHMETIC = -500;
    public static final int ERROR_ARITHMETIC_DIVIDE_BY_ZERO = -501;
    public static final int ERROR_ARITHMETIC_MATRIX_NOT_SQUARE = -502;
    public static final int ERROR_OVERFLOW = -503;
    public static final int ERROR_OPERATION = -600;
    public static final int ERROR_NOT_SUPPORTED = -601;
    public static final int ERROR_NOT_IMPLEMENTED = -602;
    public static final int ERROR_INVALID_OPERATION = -603;
    public static final int ERROR_OPERATION_CANCELED = -604;
    public static final int ERROR_TIMEOUT = -605;
    public static final int ERROR_IO = -700;
    public static final int ERROR_IO_DEVICE_BUSY = -701;
    public static final int ERROR_IO_DEVICE_NOT_ACTIVE = -702;
    public static final int ERROR_IO_DEVICE_SENSOR_FAILED = -703;
    public static final int ERROR_IO_DEVICE_NOT_CALIBRATED = -704;
    public static final int ERROR_IO_DEVICE_AUTHENTICATION_FAILED = -705;
    public static final int ERROR_IO_NO_DEVICES = -706;
    public static final int ERROR_IO_COMMAND_FAILED = -707;
    public static final int ERROR_IO_COMMAND_RESPONSE_FAILED = -708;
    public static final int ERROR_IO_COMMUNICATION_FAILED = -709;
    public static final int ERROR_IO_DATA_FIELD_INVALID = -710;
    public static final int ERROR_IO_DATA_LENGTH_INVALID = -711;
    public static final int ERROR_IO_PARAMETER_FIELD_INVALID = -712;
    public static final int ERROR_IO_DCA = -713;
    public static final int ERROR_IO_MCU = -714;
    public static final int ERROR_IO_OPERATION_CONDITIONS_INVALID = -715;
    public static final int ERROR_IO_UNKNOWN_COMMAND = -716;
    public static final int ERROR_IO_FLASH = -717;
    public static final int ERROR_IO_SOCKET = -718;
    public static final int ERROR_EXTERNAL = -800;
    public static final int ERROR_EXTERNAL_WIN32 = -801;
    public static final int ERROR_EXTERNAL_SYS = -802;
    public static final int ERROR_EXTERNAL_CPP = -803;
    public static final int ERROR_EXTERNAL_DOTNET = -804;
    public static final int ERROR_EXTERNAL_JVM = -805;
    public static final int ERROR_EXTERNAL_COM = -806;

    private static native int NBErrorsGetMessageA(int paramInt, byte[] paramArrayOfByte, NBUInt paramNBUInt,
            NBUIntByReference paramNBUIntByReference);

    private static native void NBErrorsClear();

    private static native int NBErrorsSetLastA(int paramInt1, String paramString1, int paramInt2, String paramString2,
            int paramInt3);

    public static boolean isFailed(int result) {
        /* 99 */ return result != 0;
    }

    public static boolean isSucceeded(int result) {
        /* 103 */ return !isFailed(result);
    }

    public static String getMessage(int code) {
        /* 107 */ NBUIntByReference valueLength = new NBUIntByReference();
        /* 108 */ if (isSucceeded(NBErrorsGetMessageA(code, null, new NBUInt(0L), valueLength))) {
            /* 110 */ byte[] szMessage = new byte[valueLength.getValue().intValue() + 1];
            /* 111 */ if (isSucceeded(
                    NBErrorsGetMessageA(code, szMessage, new NBUInt(szMessage.length), valueLength))) {
                /* 112 */ return Native.toString(szMessage);
            }
        }
        /* 115 */ return "";
    }

    public static int requireNoErr(int result) {
        /* 119 */ if (isFailed(result)) {
            /* 120 */ raiseError(result);
        }
        /* 122 */ return result;
    }

    public static int setLast(Throwable error) {
        /* 126 */ if (error == null)
            throw new NullPointerException("error");
        /* 127 */ Class errorType = error.getClass();

        /* 129 */ int externalErrorCode = 0;
        /* 130 */ String message = error.getMessage();
        int code;
        /* 131 */ if ((error instanceof NBThrowable)) {
            /* 132 */ code = ((NBThrowable) error).getCode();
        } else {
            /* 136 */ if ((Throwable.class.equals(errorType)) || (Exception.class.equals(errorType))
                    || (Error.class.equals(errorType)) || (RuntimeException.class.equals(errorType))) {
                /* 137 */ code = -100;
            } else {
                /* 138 */ if (IllegalArgumentException.class.isAssignableFrom(errorType)) {
                    /* 139 */ code = -200;
                } else {
                    /* 140 */ if (NullPointerException.class.isAssignableFrom(errorType)) {
                        /* 141 */ code = -201;
                    } else {
                        /* 142 */ if (IndexOutOfBoundsException.class.isAssignableFrom(errorType)) {
                            /* 143 */ code = -205;
                        } else {
                            /* 144 */ if (OutOfMemoryError.class.isAssignableFrom(errorType)) {
                                /* 145 */ code = -401;
                            } else {
                                /* 146 */ if (ArithmeticException.class.isAssignableFrom(errorType)) {
                                    /* 147 */ code = -500;
                                } else {
                                    /* 148 */ if (UnsupportedOperationException.class.isAssignableFrom(errorType)) {
                                        /* 149 */ code = -601;
                                    } else {
                                        /* 150 */ if (IllegalStateException.class.isAssignableFrom(errorType)) {
                                            /* 151 */ code = -603;
                                        } else {
                                            /* 152 */ if (CancellationException.class.isAssignableFrom(errorType)) {
                                                /* 153 */ code = -604;
                                            } else {
                                                /* 154 */ if (TimeoutException.class.isAssignableFrom(errorType)) {
                                                    /* 155 */ code = -605;
                                                } else {
                                                    /* 156 */ if (IOException.class.isAssignableFrom(errorType)) {
                                                        /* 157 */ code = -700;
                                                    } else {
                                                        /* 158 */ if (ClassCastException.class
                                                                .isAssignableFrom(errorType))
                                                            /* 159 */ code = -402;
                                                        else
                                                            /* 161 */ code = -805;
                                                    }
                                                }
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        /* 164 */ NBErrorsSetLastA(code, message, externalErrorCode, formatStackTrace(error.getStackTrace()), 16777216);
        /* 165 */ return code;
    }

    private static void raiseError(int error) {
        /* 173 */ if (isSucceeded(error))
            throw new IllegalArgumentException("error is succeeded");
        /* 174 */ getLast(error);
    }

    private static String formatStackTrace(StackTraceElement[] trace) {
        /* 178 */ StringBuilder sb = new StringBuilder(256);
        /* 179 */ sb.append('\n');
        /* 180 */ for (StackTraceElement element : trace) {
            /* 181 */ sb.append("\tat ").append(element.toString()).append("\n");
        }
        /* 183 */ return sb.toString();
    }

    private static Throwable getLast(int code, String message) {
        /* 187 */ if (isSucceeded(code)) {
            /* 188 */ return null;
        }
        /* 190 */ switch (code) {
            case -100:
                /* 192 */ return new NBRuntimeException(code, message);
            case -300:
            case -204:
            case -203:
            case -202:
            case -200:
                /* 198 */ return new NBIllegalArgumentException(code, message);
            case -205:
                /* 200 */ return new NBIndexOutOfBoundsException(code, message);
            case -201:
                /* 202 */ return new NBNullPointerException(code, message);
            case -401:
            case -400:
                /* 205 */ return new NBOutOfMemoryError(code, message);
            case -402:
                /* 207 */ return new NBClassCastException(code, message);
            case -503:
            case -502:
            case -501:
            case -500:
                /* 212 */ return new NBArithmeticException(code, message);
            case -602:
            case -601:
            case -600:
                /* 216 */ return new NBUnsupportedOperationException(code, message);
            case -603:
                /* 218 */ return new NBIllegalStateException(code, message);
            case -604:
                /* 220 */ return new NBCancellationException(code, message);
            case -605:
                /* 222 */ return new NBTimeoutException(code, message);
            case -718:
            case -717:
            case -716:
            case -715:
            case -714:
            case -713:
            case -712:
            case -711:
            case -710:
            case -709:
            case -708:
            case -707:
            case -706:
            case -705:
            case -704:
            case -703:
            case -702:
            case -701:
            case -700:
                /* 242 */ return new NBIOException(code, message);
            case -806:
            case -805:
            case -804:
            case -803:
            case -802:
            case -801:
            case -800:
                /* 250 */ return new ExternalException(code, message);
        }
        /* 252 */ return new NextBiometricsException(code, message);
    }

    private static void getLast(int error) {
        /* 258 */ Throwable th = getLast(error, getMessage(error));
        /* 259 */ if (th == null)
            /* 260 */ return;
        /* 261 */ if ((th instanceof RuntimeException))
            /* 262 */ throw ((RuntimeException) th);
        /* 263 */ if ((th instanceof Error)) {
            /* 264 */ throw ((Error) th);
        }
        /* 266 */ throw new RuntimeException(th.getMessage(), th);
    }

    static {
        try {
            /* 81 */ Native.register(NBErrors.class, NBSystemLibrary.NATIVE_LIBRARY);
        } catch (Throwable e) {
            /* 83 */ throw new ExceptionInInitializerError(e);
        }
    }
}
